/*
 * Copyright (c) 2013 Adobe Systems Incorporated. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 */
/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4, maxerr: 50, continue: true, newcap:true, regexp:true */
/*global DW_HTMLSimpleDOM, window, DW_LIVEEDIT_CONSTANTS, $dwJQuery, DW_HTMLDOMDiff, DW_RemoteFunctions*/
/*Global Vars*/
var DW_LiveEdit_IDString = DW_LIVEEDIT_CONSTANTS.DWUniqueId + '="';
var DW_LiveEdit_IDManagerInstance = null;
var DW_LiveEdit_LiveSource = "";
var DW_LiveEdit_CurrentSimpleDOM = null;
var DW_LiveEdit_ReplacedDOM = null;

function DW_LiveEdit_IDManager() {
    
    this.DW_LiveEdit_CurrentID = 0;
    this.DW_LiveEdit_IDMap = {};
    this.DW_LiveEdit_IDMap_temp = {};
    this.DW_LiveEdit_ID2dwIDMap = {};
}    

DW_LiveEdit_IDManager.prototype.getID = function(newTag, delta) {
    var ret = -1;
    delta = delta || 0;
    var intKey = parseInt(newTag.start) + delta;
    
    if(this.DW_LiveEdit_IDMap[intKey] !== undefined && this.DW_LiveEdit_IDMap[intKey].tag === newTag.tag) {
        ret = this.DW_LiveEdit_IDMap[intKey].id;
    }
    /*
    else if(this.DW_LiveEdit_IDMap_temp[intKey] !== undefined && this.DW_LiveEdit_IDMap_temp[intKey].tag === newTag.tag) {
        //previously we used to check just the character offset and tagName to reuse ids
        //But if we reuse the IDs, then chances are DomDiff will try to move element
        //This can have adverse affects, so commenting out for now
        
        ret = this.DW_LiveEdit_IDMap_temp[intKey].id;
        this.DW_LiveEdit_IDMap[intKey] = this.DW_LiveEdit_IDMap_temp[intKey];    
              
    }
    */
    return ret;
}
    
DW_LiveEdit_IDManager.prototype.updateID = function(startPivot, endPivot, diff) {
    this.DW_LiveEdit_IDMap_temp = {};
    tempMap = {};
    
    for(key in this.DW_LiveEdit_IDMap) {
        var intKey = parseInt(key);
        if(intKey < startPivot) {
            tempMap[intKey] = this.DW_LiveEdit_IDMap[intKey];
        }
        else if(intKey >= endPivot) {
            tempMap[intKey + diff] = this.DW_LiveEdit_IDMap[intKey];
        }
        else {
            this.DW_LiveEdit_IDMap_temp[intKey] = this.DW_LiveEdit_IDMap[intKey];
        }
    }
    this.DW_LiveEdit_IDMap = tempMap;
}
    
DW_LiveEdit_IDManager.prototype.clearTemp = function() {
    this.DW_LiveEdit_IDMap_temp = {};
}
    
DW_LiveEdit_IDManager.prototype.setID = function(newTag, delta) {
    var ret = -1;
    delta = delta || 0;
    var intKey = parseInt(newTag.start) + delta;
    
    if(intKey !== undefined && newTag.tag !== undefined) {
        this.DW_LiveEdit_IDMap[intKey] = {'id':++this.DW_LiveEdit_CurrentID , 'tag':newTag.tag};
        ret = this.DW_LiveEdit_CurrentID;
    }
    return ret;
}

//Utility function to find the DW_LiveEdit_ID of a tag given the start offset of the tag
//it searches of the immediate liveedit-id attribute and extracts the attribute value
function DW_LiveEdit_findID(domString, startOffset) {
    'use strict';
    var index = domString.indexOf('>', startOffset);
    var subString = domString.slice(startOffset, index);
    var idIndex = subString.indexOf(DW_LiveEdit_IDString);
    if (idIndex !== -1) {
        var start = idIndex + DW_LiveEdit_IDString.length;
        var end = subString.indexOf('"', start);
        return subString.slice(start, end);
    } else {
        console.error('Could not find the id at in the string: ' + subString);
        return ++(DW_LiveEdit_IDManagerInstance.DW_LiveEdit_CurrentID);
    }
}

//called as soon as the doc is loaded, it takes in the entire code source as parameter
//this acts as a base upon which all edits are applied
//here we construct the SimpleDOM wrt to the source and keep it
function DW_LiveEdit_SetupDoc(source) {
    'use strict';
    DW_LiveEdit_LiveSource = source;
    DW_LiveEdit_IDManagerInstance = new DW_LiveEdit_IDManager();
    
    function findID(newTag) {
        var dwID = DW_LiveEdit_findID(DW_LiveEdit_LiveSource, newTag.start);
        var newID = DW_LiveEdit_IDManagerInstance.setID(newTag, 0);
        if(dwID)
            DW_LiveEdit_IDManagerInstance.DW_LiveEdit_ID2dwIDMap[newID] = dwID;
        return newID;
    }

    var Builder = DW_HTMLSimpleDOM().Builder;
    var builder = new Builder(DW_LiveEdit_LiveSource, 0, null);
    builder.getID = findID;
    var markCache = {};
    DW_LiveEdit_CurrentSimpleDOM = builder.build(false, markCache);

    if (!DW_LiveEdit_CurrentSimpleDOM) {
        console.error('Could not create SImpleDOM structure mostly cause of malformed HTML');
    }
}

//Upon an edit, we pass the startOffset and endOffset of the edit wrt to the base string prior to the edit
//also we pass the new string if any. First we construct the new string and its corresponding SimpleDOM
//compare it with the earlier DOM to generate the edits and finally pass it along
function DW_LiveEdit_DoEdit(startOffset, endOffset, editString, shouldOnlyUpdate) {
    function findID(newTag) {
        var dwID = DW_LiveEdit_findID(newString, newTag.start);
        var id = DW_LiveEdit_IDManagerInstance.getID(newTag, 0);
        if( id === -1) {
            id = DW_LiveEdit_IDManagerInstance.setID(newTag, 0);
        }
        if(shouldOnlyUpdate) {
            DW_LiveEdit_IDManagerInstance.DW_LiveEdit_ID2dwIDMap[id] = dwID;
        }
        return id;
    }

    function findPartialDOM() {
        //find the tag closest to startOffset and endOffset        
        function getParentForOffset(offset) {
            var nodeMap = DW_LiveEdit_CurrentSimpleDOM.nodeMap;
            var retVal = null;

            var idString = "";
            var dwIDIndex = DW_LiveEdit_LiveSource.lastIndexOf(DW_LiveEdit_IDString, offset);
            var tagIndex = dwIDIndex !== -1 ? DW_LiveEdit_LiveSource.lastIndexOf('<', dwIDIndex) : -1;
            if (tagIndex !== -1) {
                if(DW_LiveEdit_IDManagerInstance.DW_LiveEdit_IDMap[tagIndex]) {
                    idString = DW_LiveEdit_IDManagerInstance.DW_LiveEdit_IDMap[tagIndex].id;
                }
                else if(DW_LiveEdit_IDManagerInstance.DW_LiveEdit_IDMap_temp[tagIndex]) {
                    idString = DW_LiveEdit_IDManagerInstance.DW_LiveEdit_IDMap_temp[tagIndex].id;
                }                
            }

            if (idString !== "") {
                var parentNode = nodeMap[idString];
                while (parentNode) {
					var start = parentNode.start + parentNode.tag.length; 
					var end = parentNode.end - (parentNode.tag.length + 3); //considering the entire endtag
				
                    if (start < offset && end > offset) {
                        retVal = parentNode;
                        break;
                    } else {
                        parentNode = parentNode.parent;
                    }
                }
            }

            return retVal;
        }

        function getOrphanClosingTagCount(str) {
           /*
            imagine the edit being addition of '</div></div><div><div>' string, in this case the parent to be reconstructed goes to level higher
            so in this function we will figure out if we are closing parent tags and accordingly go higher in order.
            for this we will write a stack to figure out if we closing tags that we did not open
            
            Eg: src="1.png"><img src="2.png"></span><br><span>asdfasdfasdf</span>
            The regex will create >,<img,>,</span,>,<br,>,<span,>,</span,>
            
            The first for loop will try fixing up '>', if an opening tag is found before '>', we will delete it
            hence after first loop: >,<img,</span,<br,<span,</span
            
            Next loop tries to see if there are complete elements, i.e. both open and end, it removes them
            hence after 2nd loop: >,<img,</span,<br
            
            Third loop sees potential tags that string is closing but not opening, in this case they are > & </span
            
            */
            var count = 0;
            if (str && str !== "") {
                var re = new RegExp(/<\/?[^\s>]+|>/g);
                var matchArray = str.match(re);

                if (matchArray) {
                    var index;
                    for(index = 1; index < matchArray.length; ++index) {
                        var matchStr = matchArray[index];
                        var matchPrev = matchArray[index-1];
                        if(matchStr === ">" && matchPrev.charAt(0) === '<') {
                            matchArray.splice(index,1);
                            --index;
                        }
                    }                
                
                    for (index = 1; index < matchArray.length; ++index) {
                        if (matchArray[index].indexOf("/") !== -1) {
                            var match1 = matchArray[index - 1].toLowerCase();
                            var match2 = matchArray[index].toLowerCase().replace(/\//g, '');

                            if (match1 === match2) {
                                matchArray.splice(index - 1, 2);
                                --index;
                            }
                        }
                    }

                    for (index = 0; index < matchArray.length; ++index) {
                        var match = matchArray[index];
                        if (match.indexOf("/") !== -1 || match === ">") {
                            ++count;
                        }
                    }
                }
            }
            return count;
        }

        var startParent = getParentForOffset(startOffset),
            endParent = getParentForOffset(endOffset),
            commonParent = null;

        //once the tags are found, go up the DOM heirarchy to find a common parent
        while (startParent && endParent) {
            if (startParent === endParent) {
                commonParent = endParent;
                var mismatch = getOrphanClosingTagCount(editString);
                while (commonParent && mismatch > 0) {
                    commonParent = commonParent.parent;
                    --mismatch;
                }
                break;
            } else if (startParent.start < endParent.start) {
                if (endParent.parent) {
                    endParent = endParent.parent;
                } else {
                    break;
                }
            } else {
                if (startParent.parent) {
                    startParent = startParent.parent;
                } else {
                    break;
                }
            }
        }

        return commonParent;
    }

    function buildNodeMap(root) {
        var nodeMap = {};

        function walk(node) {
            if (node.tagID) {
                nodeMap[node.tagID] = node;
            }
            if (node.isElement()) {
                node.children.forEach(walk);
            }
        }

        walk(root);
        root.nodeMap = nodeMap;
    }

    function handleDeletions(nodeMap, oldSubtreeMap, newSubtreeMap) {
        var deletedIDs = [];
        Object.keys(oldSubtreeMap).forEach(function (key) {
            if (!newSubtreeMap.hasOwnProperty(key)) {
                deletedIDs.push(key);
                delete nodeMap[key];
            }
        });
    }

    //it adds the delta to start and end offsets if the offset is greater than the pivot
    //keep pivot=0 if you want it to apply for all offsets
    function updateOffsetsWithDelta(root, delta, pivot) {
        var nodeMap = root.nodeMap;
        var nodeID;
        for (nodeID in nodeMap) {
            var node = nodeMap[nodeID];
            if (node.start !== undefined && node.start >= pivot) {
                node.start += delta;
            }
            if (node.end !== undefined && node.end >= pivot) {
                node.end += delta;
            }
        }
    }

    if (!DW_LiveEdit_CurrentSimpleDOM || DW_LiveEdit_LiveSource.length === 0) {
        //Should never reach here, just in case we do let us refresh the browser
        // Notify Dreamweaver to put the files on server before reloading the page.
        window.DWApplyDOMEditsHasInstruction();
        return;
    }

    var replacedString = DW_LiveEdit_LiveSource.slice(startOffset, endOffset);
    var newString = DW_LiveEdit_LiveSource.slice(0, startOffset) + editString + DW_LiveEdit_LiveSource.slice(endOffset); //modified document string
    var delta = endOffset - startOffset - editString.length; //length adjustment of edit
    DW_LiveEdit_IDManagerInstance.updateID(startOffset, endOffset, -delta);
    
    var prevSubDOM = findPartialDOM();
    var diff = null,
        incrementalSuccess = false, /*bool to check if we were successful in doing an incremental update*/
        updateSuccess = false; /*bool to check if we were able to create an updated SimpleDOM structure using the current edit and apply the same*/

    if (prevSubDOM && prevSubDOM.parent) {
        function findNewID(newTag) {
            var dwID = DW_LiveEdit_findID(newSubString, newTag.start);
            var id = DW_LiveEdit_IDManagerInstance.getID(newTag, prevSubDOM.start);
            if( id === -1) {
                id = DW_LiveEdit_IDManagerInstance.setID(newTag, prevSubDOM.start);
            } 
            if(shouldOnlyUpdate) {
                DW_LiveEdit_IDManagerInstance.DW_LiveEdit_ID2dwIDMap[id] = dwID;
            } 
            return id;
        }
        
        var newSubString = newString.slice(prevSubDOM.start, prevSubDOM.end - delta);
        var parent = prevSubDOM.parent;

        var Builder = DW_HTMLSimpleDOM().Builder;
        var builder = new Builder(newSubString, 0, null);
        builder.getID = findNewID;
        var markCache = {};
        var nextSubDOM = builder.build(false, markCache);

        if (parent && nextSubDOM) {
            var childIndex = parent.children.indexOf(prevSubDOM);
            if (childIndex === -1) {
                // This should never happen...
                console.error("couldn't locate old subtree in tree");
            } else {
                // Swap the new subtree in place of the old subtree.
                prevSubDOM.parent = null;
                nextSubDOM.parent = parent;
                parent.children[childIndex] = nextSubDOM;

                //update offsets as per the edit
                updateOffsetsWithDelta(DW_LiveEdit_CurrentSimpleDOM, -delta, startOffset);
                updateOffsetsWithDelta(nextSubDOM, prevSubDOM.start, 0);

                // Overwrite any node mappings in the parent DOM with the
                // mappings for the new subtree. We keep the nodeMap around
                // on the new subtree so that the differ can use it later.
                $dwJQuery.extend(DW_LiveEdit_CurrentSimpleDOM.nodeMap, nextSubDOM.nodeMap);

                // Build a local nodeMap for the old subtree so the differ can
                // use it.
                buildNodeMap(prevSubDOM);

                // Clean up the info for any deleted nodes that are no longer in
                // the new tree.
                handleDeletions(DW_LiveEdit_CurrentSimpleDOM.nodeMap, prevSubDOM.nodeMap, nextSubDOM.nodeMap);

                // Update the signatures for all parents of the new subtree.
                var curParent = parent;
                while (curParent) {
                    curParent.update();
                    curParent = curParent.parent;
                }

                if (!shouldOnlyUpdate) {
                    diff = DW_HTMLDOMDiff().domdiff(prevSubDOM, nextSubDOM);
                }
                incrementalSuccess = true;
                updateSuccess = true;
                DW_LiveEdit_ReplacedDOM = prevSubDOM;
            }
        }
    }

    //incase we couldn't success fully apply an inceremental update, rebuild the DOM and do a complete diff though exoensive
    if (!incrementalSuccess) {
        
        var domStart = DW_LiveEdit_CurrentSimpleDOM.start + DW_LiveEdit_CurrentSimpleDOM.tag.length;  //considering start of tag including name
        var domEnd = DW_LiveEdit_CurrentSimpleDOM.end - (DW_LiveEdit_CurrentSimpleDOM.tag.length + 3); // considering entire end tags, 3 represents the characters "</>"
        if(startOffset < domStart || endOffset > domEnd) {
            //This happens when the source is broken and we are having a tree that does not contain things fully
            //In this case we can find edits of children of the main tree only
            // Notify Dreamweaver to put the files on server before reloading the page.
            window.DWApplyDOMEditsHasInstruction();
            return;
        }    
    
        var Builder = DW_HTMLSimpleDOM().Builder;
        var builder = new Builder(newString, 0, null);
        builder.getID = findID;
        var markCache = {};
        var newDOM = builder.build(false, markCache);

        if (newDOM) {
            if (!shouldOnlyUpdate) {
                diff = DW_HTMLDOMDiff().domdiff(DW_LiveEdit_CurrentSimpleDOM, newDOM);
            }
            DW_LiveEdit_ReplacedDOM = DW_LiveEdit_CurrentSimpleDOM;
            DW_LiveEdit_CurrentSimpleDOM = newDOM;
            updateSuccess = true;
        }
    }

    if (diff) {
        //We successfully generated edits implies update has been successful so far
        updateSuccess = DW_RemoteFunctions().applyDOMEdits(diff); //we will get a false we were unable to find the element upon which the edit had to be done
    }

    if (updateSuccess) {
        DW_LiveEdit_IDManagerInstance.clearTemp();
        DW_LiveEdit_LiveSource = newString; // we were able to update our structures hence it is safe to update the source string
        DW_LiveEdit_ReplacedDOM = null;
        //recreate the Editable Regions HUD in live view since the DOM has changed
        if (window.reInitializeEditableRegions) {
            window.reInitializeEditableRegions();
        }
    } else {
        if( DW_RemoteFunctions().applyDOMEditsHasInstruction() ) {
            DW_RemoteFunctions().resetApplyDOMEditsHasInstruction();
			// Notify Dreamweaver that the partial refresh failed because of a php edit or something
            window.DWApplyDOMEditsHasInstruction();
        }
        else {
            // Notify Dreamweaver to put the files on server before reloading the page.
            window.DWApplyDOMEditsHasInstruction();
        }
    }
}

function DW_GetRectOfElementWithGivenID(id) {
    var results = document.querySelectorAll("[" + DW_LIVEEDIT_CONSTANTS.DWUniqueId + "='" + id + "']");
    var elem = results && results[0];
    if(elem) {
        window.ScrollElementIntoViewIfNeeded(id, "");
        var rect = elem.getBoundingClientRect();
        if(rect) {
            var retString = rect.left + ',' + rect.top + ',' + rect.right + ',' + rect.bottom;
            return retString;
        }
    }
    return '';
}
